<?php

namespace App\Console\Commands;

use App\Models\UserSubscription;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class CheckSubscriptionExpiry extends Command
{
    protected $signature = 'subscriptions:check-expiry';
    protected $description = 'Check and update expired subscriptions, send reminders for expiring ones';

    public function handle()
    {
        // Mark expired subscriptions
        $expiredCount = UserSubscription::where('status', 'active')
            ->where('expires_at', '<=', now())
            ->update(['status' => 'expired']);

        $this->info("Marked {$expiredCount} subscriptions as expired.");
        Log::info("Subscription expiry check: {$expiredCount} marked as expired");

        // Find subscriptions expiring soon (within 24 hours) that haven't been reminded
        $expiringSoon = UserSubscription::with(['user', 'plan'])
            ->where('status', 'active')
            ->where('expires_at', '>', now())
            ->where('expires_at', '<=', now()->addHours(24))
            ->where('reminder_sent', false)
            ->get();

        foreach ($expiringSoon as $subscription) {
            // Mark reminder as sent
            $subscription->update(['reminder_sent' => true]);
            
            // Log for now - you can add email/notification here
            Log::info("Subscription expiring soon", [
                'user_id' => $subscription->user_id,
                'user_email' => $subscription->user->email ?? 'N/A',
                'plan' => $subscription->plan->name ?? 'N/A',
                'expires_at' => $subscription->expires_at,
            ]);
        }

        $this->info("Sent {$expiringSoon->count()} expiry reminders.");
        Log::info("Subscription expiry reminders sent: {$expiringSoon->count()}");

        return Command::SUCCESS;
    }
}
