<?php

// app/Http/Controllers/CorrectScoreGameController.php

namespace App\Http\Controllers;

use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Models\CorrectScoreGame;
use App\Models\CorrectScoreGameGroup;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CorrectScoreGameController extends Controller
{
    // Cache duration in minutes - optimized for Redis
    private const CACHE_TTL = 60; // 1 hour
    private const CACHE_TTL_SHORT = 30; // 30 minutes for date-specific queries

    public function index()
    {
        // Cache the query with eager loading
        $correctScoreGameGroups = Cache::remember('correct_score_groups.index', self::CACHE_TTL, function () {
            return CorrectScoreGameGroup::with(['games' => function($query) {
                $query->select('id', 'correct_score_game_group_id', 'date_time', 'home_team', 'away_team', 'league', 'option', 'odds', 'status');
            }])
                ->select('id', 'price', 'selling', 'status', 'sporty_booking', 'onex_booking', 'msport_booking', 'betway_booking', 'created_at', 'updated_at')
                ->orderBy('created_at', 'desc')
                ->limit(50) // Add reasonable limit
                ->get();
        });

        return response()->json($correctScoreGameGroups, 200);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'price' => 'required|numeric',
            'selling' => 'required|boolean',
            'games' => 'required|array',
            'sporty_booking' => 'nullable|string',
            'onex_booking' => 'nullable|string',
            'msport_booking' => 'nullable|string',
            'betway_booking' => 'nullable|string',
            'created_at' => 'nullable|date',
            'games.*.date_time' => 'required|date',
            'games.*.home_team' => 'required|string|max:255',
            'games.*.away_team' => 'required|string|max:255',
            'games.*.league' => 'required|string|max:255',
            'games.*.option' => 'required|string|max:255',
            'games.*.odds' => 'required|numeric',
        ]);

        DB::beginTransaction();
        try {
            $correctScoreGameGroup = CorrectScoreGameGroup::create([
                'price' => $validated['price'],
                'selling' => $validated['selling'],
                'sporty_booking' => $validated['sporty_booking'] ?? null,
                'onex_booking' => $validated['onex_booking'] ?? null,
                'msport_booking' => $validated['msport_booking'] ?? null,
                'betway_booking' => $validated['betway_booking'] ?? null,
                'created_at' => $validated['created_at'] ?? Carbon::now(),
            ]);

            // Bulk insert games for better performance
            $games = $validated['games'];
            $timestamp = now();
            $bulkGames = array_map(function($game) use ($correctScoreGameGroup, $timestamp) {
                $game['correct_score_game_group_id'] = $correctScoreGameGroup->id;
                $game['created_at'] = $timestamp;
                $game['updated_at'] = $timestamp;
                return $game;
            }, $games);

            CorrectScoreGame::insert($bulkGames);

            // Fetch the created games for response
            $createdGames = CorrectScoreGame::where('correct_score_game_group_id', $correctScoreGameGroup->id)->get();

            DB::commit();

            // Clear relevant caches
            $this->clearCorrectScoreCaches();

            return response()->json(['group' => $correctScoreGameGroup, 'games' => $createdGames], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating correct score game group: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create group and games', 'details' => $e->getMessage()], 500);
        }
    }

    public function show($id)
    {
        // Cache individual group lookups
        $cacheKey = "correct_score_group.{$id}";
        $correctScoreGameGroup = Cache::remember($cacheKey, self::CACHE_TTL, function () use ($id) {
            return CorrectScoreGameGroup::with(['games' => function($query) {
                $query->select('id', 'correct_score_game_group_id', 'date_time', 'home_team', 'away_team', 'league', 'option', 'odds', 'status');
            }])->find($id);
        });

        if (is_null($correctScoreGameGroup)) {
            return response()->json(['message' => 'Group Not Found'], 404);
        }

        return response()->json($correctScoreGameGroup, 200);
    }

    public function update(Request $request, $id)
    {
        $correctScoreGameGroup = CorrectScoreGameGroup::find($id);
        if (is_null($correctScoreGameGroup)) {
            return response()->json(['message' => 'Group Not Found'], 404);
        }

        DB::beginTransaction();
        try {
            // Update group fields
            $correctScoreGameGroup->update($request->only('price', 'selling', 'msport_booking', 'onex_booking', 'betway_booking', 'sporty_booking', 'created_at'));

            // Handle nested games array
            if ($request->has('games')) {
                $games = $request->input('games');

                // Delete existing games in one query
                CorrectScoreGame::where('correct_score_game_group_id', $id)->delete();

                // Bulk insert new games
                $timestamp = now();
                $bulkGames = array_map(function($game) use ($id, $timestamp) {
                    $game['correct_score_game_group_id'] = $id;
                    $game['created_at'] = $timestamp;
                    $game['updated_at'] = $timestamp;
                    return $game;
                }, $games);

                CorrectScoreGame::insert($bulkGames);
            }

            // Load the updated games relationship
            $correctScoreGameGroup->load('games');

            DB::commit();

            // Clear caches
            Cache::forget("correct_score_group.{$id}");
            $this->clearCorrectScoreCaches();

            return response()->json($correctScoreGameGroup, 200);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating correct score game group: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update group and games', 'details' => $e->getMessage()], 500);
        }
    }

    public function destroy($id)
    {
        $correctScoreGameGroup = CorrectScoreGameGroup::find($id);
        if (is_null($correctScoreGameGroup)) {
            return response()->json(['message' => 'Group Not Found'], 404);
        }

        DB::beginTransaction();
        try {
            $correctScoreGameGroup->delete(); // This will cascade delete games if foreign key is set
            DB::commit();

            // Clear caches
            Cache::forget("correct_score_group.{$id}");
            $this->clearCorrectScoreCaches();

            return response()->json(['message' => 'Group Deleted'], 204);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting correct score game group: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete group', 'details' => $e->getMessage()], 500);
        }
    }

    public function getCorrectScoreGamesByDateRange(Request $request)
    {
        $date = $request->query('date');
        if (!$date) {
            return response()->json(['message' => 'Date is required'], 400);
        }

        $parsedDate = Carbon::parse($date)->format('Y-m-d');

        // Cache with date-specific key
        $cacheKey = "correct_score_groups_by_date.{$parsedDate}";
        $correctScoreGameGroups = Cache::remember($cacheKey, self::CACHE_TTL_SHORT, function () use ($date) {
            return CorrectScoreGameGroup::with(['games' => function($query) {
                $query->select('id', 'correct_score_game_group_id', 'date_time', 'home_team', 'away_team', 'league', 'option', 'odds', 'status');
            }])
                ->select('id', 'price', 'selling', 'status', 'sporty_booking', 'onex_booking', 'msport_booking', 'betway_booking', 'created_at', 'updated_at')
                ->whereDate('created_at', $date)
                ->orderBy('created_at', 'desc')
                ->get();
        });

        return response()->json($correctScoreGameGroups, 200);
    }

    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'nullable|in:Won,Lost',
            'type' => 'required|in:group,individual',
        ]);

        DB::beginTransaction();
        try {
            if ($request->type === 'group') {
                $gameGroup = CorrectScoreGameGroup::findOrFail($id);
                $gameGroup->status = $request->status;
                $gameGroup->save();

                // Clear cache for this group
                Cache::forget("correct_score_group.{$id}");
                $this->clearCorrectScoreCaches();

                DB::commit();
                return response()->json(['message' => 'Group game status updated successfully']);

            } else if ($request->type === 'individual') {
                $individualGame = CorrectScoreGame::findOrFail($id);
                $individualGame->status = $request->status;
                $individualGame->save();

                // Clear cache for the parent group
                if ($individualGame->correct_score_game_group_id) {
                    Cache::forget("correct_score_group.{$individualGame->correct_score_game_group_id}");
                }
                $this->clearCorrectScoreCaches();

                DB::commit();
                return response()->json(['message' => 'Individual game status updated successfully']);
            }

            return response()->json(['message' => 'Invalid request'], 400);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating correct score game status: ' . $e->getMessage());
            return response()->json(['message' => 'Error updating status'], 500);
        }
    }

    /**
     * Clear all correct score-related caches
     */
    private function clearCorrectScoreCaches()
    {
        Cache::forget('correct_score_groups.index');

        // Clear date-specific caches for the last 7 days
        for ($i = 0; $i < 7; $i++) {
            $date = Carbon::now()->subDays($i)->format('Y-m-d');
            Cache::forget("correct_score_groups_by_date.{$date}");
        }

        // If using cache tags (Redis) - only use if Redis supports it
        if (Cache::getStore() instanceof \Illuminate\Cache\RedisStore) {
            try {
                Cache::tags(['correct_score_groups'])->flush();
            } catch (\Exception $e) {
                // Tags might not be supported, continue without error
            }
        }
    }
}
