<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use Illuminate\Support\Facades\Mail;
use App\Mail\PasswordResetOtpMail;

class PasswordResetController extends Controller
{
    public function sendResetOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:users,email',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 422);
        }

        $user = User::where('email', $request->email)->first();

        // Generate a random 6-digit OTP code
        $otp = mt_rand(100000, 999999);

        // Store OTP with an expiry time (e.g., 10 minutes)
        Cache::put('password-reset-otp-' . $user->email, $otp, 600);

        // Send OTP to user's email
        Mail::to($user->email)->send(new PasswordResetOtpMail($otp));

        return response()->json(['message' => 'OTP sent to your email.']);
    }

    public function resetPasswordWithOtp(Request $request)
    {
        //Log::info($request->all());
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:users,email',
            'otp' => 'required|numeric',
            'password' => 'required|confirmed|min:6',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 422);
        }

        $user = User::where('email', $request->email)->first();

        $storedOtp = Cache::get('password-reset-otp-' . $user->email);
       // Log::info($storedOtp);
        if ((int)$storedOtp !== (int)$request->otp) {
            return response()->json(['error' => 'Invalid OTP.'], 400);
        }

        // Update user's password
        $user->password = Hash::make($request->password);
        $user->save();

        // Clear the OTP from cache
        Cache::forget('password-reset-otp-' . $user->email);

        return response()->json(['message' => 'Password reset successful.']);
    }
}
