<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class UserSubscription extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'subscription_plan_id',
        'starts_at',
        'expires_at',
        'status',
        'amount_paid',
        'payment_method',
        'payment_reference',
        'admin_notes',
        'activated_by',
        'reminder_sent',
    ];

    protected $casts = [
        'starts_at' => 'datetime',
        'expires_at' => 'datetime',
        'amount_paid' => 'decimal:2',
        'reminder_sent' => 'boolean',
    ];

    /**
     * Override status attribute to return computed status based on expiry
     */
    public function getStatusAttribute($value)
    {
        if ($value === 'cancelled') {
            return 'cancelled';
        }
        
        if ($this->expires_at && $this->expires_at <= now()) {
            return 'expired';
        }
        
        return $value;
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function plan()
    {
        return $this->belongsTo(SubscriptionPlan::class, 'subscription_plan_id');
    }

    public function activatedBy()
    {
        return $this->belongsTo(User::class, 'activated_by');
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active')
                     ->where('expires_at', '>', now());
    }

    public function scopeExpired($query)
    {
        return $query->where('status', 'expired')
                     ->orWhere('expires_at', '<=', now());
    }

    public function scopeExpiringSoon($query, $days = 1)
    {
        return $query->where('status', 'active')
                     ->where('expires_at', '>', now())
                     ->where('expires_at', '<=', now()->addDays($days));
    }

    public function isActive()
    {
        return $this->status === 'active' && $this->expires_at > now();
    }

    public function daysRemaining()
    {
        if (!$this->isActive()) {
            return 0;
        }
        return max(0, now()->diffInDays($this->expires_at, false));
    }

    public function markAsExpired()
    {
        $this->update(['status' => 'expired']);
    }
}
